from sqlalchemy.orm import Session
from sqlalchemy import select
from fastapi import HTTPException, status, Depends
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from app.models.user import User
from app.utils.auth import verify_password, get_password_hash, create_access_token, verify_token
from app.schemas.user import UserCreate, UserLogin
from app.utils.logging import log_user_action
from typing import Optional

security = HTTPBearer()


class AuthService:
    def __init__(self, db: Session):
        self.db = db

    def register_user(self, user_data: UserCreate, ip_address: Optional[str] = None) -> User:
        # Check if user already exists
        existing_user = self.db.execute(
            select(User).where((User.email == user_data.email) | (User.username == user_data.username))
        ).scalar_one_or_none()
        
        if existing_user:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST,
                detail="User with this email or username already exists"
            )

        # Create new user
        hashed_password = get_password_hash(user_data.password)
        db_user = User(
            username=user_data.username,
            email=user_data.email,
            hashed_password=hashed_password
        )
        
        self.db.add(db_user)
        self.db.commit()
        self.db.refresh(db_user)

        # Log registration
        log_user_action(
            self.db,
            db_user.id,
            "register",
            {"username": user_data.username, "email": user_data.email},
            ip_address
        )

        return db_user

    def authenticate_user(self, login_data: UserLogin, ip_address: Optional[str] = None) -> User:
        user = self.db.execute(
            select(User).where(User.email == login_data.email)
        ).scalar_one_or_none()

        if not user or not verify_password(login_data.password, user.hashed_password):
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Incorrect email or password"
            )

        # Log login
        log_user_action(
            self.db,
            user.id,
            "login",
            {"email": login_data.email},
            ip_address
        )

        return user

    def get_current_user(self, credentials: HTTPAuthorizationCredentials = Depends(security)) -> User:
        token_data = verify_token(credentials.credentials)
        user_id = token_data["user_id"]
        
        user = self.db.execute(select(User).where(User.id == user_id)).scalar_one_or_none()
        
        if user is None:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="User not found"
            )
        
        return user