from pydantic import BaseModel, Field, validator
from typing import Optional
import base64
import re


class ImageInputBase64(BaseModel):
    image_data: str = Field(..., description="Base64 encoded image data")
    filename: Optional[str] = Field(default="image.jpg", description="Original filename for logging")
    
    @validator('image_data')
    def validate_base64_image(cls, v):
        # Remove data URL prefix if present (data:image/jpeg;base64,)
        if v.startswith('data:image'):
            try:
                # Extract base64 part after comma
                v = v.split(',')[1]
            except IndexError:
                raise ValueError("Invalid data URL format")
        
        # Validate base64 format
        try:
            base64.b64decode(v, validate=True)
        except Exception:
            raise ValueError("Invalid base64 encoding")
        
        return v
    
    @validator('filename')
    def validate_filename(cls, v):
        if v:
            # Check file extension
            allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp']
            file_extension = v.lower().split('.')[-1] if '.' in v else ''
            if file_extension not in allowed_extensions:
                raise ValueError(f"File extension not supported. Allowed: {', '.join(allowed_extensions)}")
        return v or "image.jpg"


class ImageInputChoice(BaseModel):
    """
    Schema that accepts either file upload or base64 image data
    Only one of these should be provided
    """
    base64_data: Optional[ImageInputBase64] = Field(None, description="Base64 image data")
    
    @validator('base64_data')
    def validate_input_choice(cls, v, values):
        if not v:
            raise ValueError("base64_data is required")
        return v