from fastapi import APIRouter, Depends, UploadFile, File, Request
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
from sqlalchemy.orm import Session
from app.database import get_db
from app.schemas.face_detection import FaceDetectionResponse
from app.schemas.image_input import ImageInputBase64
from app.services.auth_service import AuthService
from app.services.face_detection_service import FaceDetectionService
from app.utils.logging import log_user_action
from app.models.user import User
import uuid

router = APIRouter(prefix="/api/face", tags=["face detection"])

face_service = FaceDetectionService()
security = HTTPBearer()


def get_current_user(
    credentials: HTTPAuthorizationCredentials = Depends(security),
    db: Session = Depends(get_db)
):
    auth_service = AuthService(db)
    return auth_service.get_current_user(credentials)


@router.post("/detect", response_model=FaceDetectionResponse)
async def detect_faces_file(
    request: Request,
    file: UploadFile = File(...),
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """Detect faces from uploaded file"""
    # Validate file
    face_service.validate_image_file(file.filename, file.size)
    
    # Read file content
    file_content = await file.read()
    
    # Detect faces
    faces = face_service.detect_faces_from_bytes(file_content)
    
    # Generate transaction ID and log
    transaction_id = _log_detection_and_return_id(
        db, current_user, request, 
        filename=file.filename,
        file_size=len(file_content),
        faces=faces,
        input_type="file_upload"
    )
    
    return FaceDetectionResponse(
        transaction_id=transaction_id,
        face_count=len(faces),
        faces=faces,
        success=True,
        message=f"Successfully detected {len(faces)} face(s) from uploaded file"
    )


@router.post("/detect-base64", response_model=FaceDetectionResponse)
def detect_faces_base64(
    request: Request,
    image_data: ImageInputBase64,
    current_user: User = Depends(get_current_user),
    db: Session = Depends(get_db)
):
    """Detect faces from base64 encoded image"""
    # Validate base64 image
    face_service.validate_base64_image(image_data.image_data)
    
    # Detect faces
    faces = face_service.detect_faces_from_base64(image_data.image_data)
    
    # Calculate approximate file size for logging
    estimated_size = int(len(image_data.image_data) * 0.75)
    
    # Generate transaction ID and log
    transaction_id = _log_detection_and_return_id(
        db, current_user, request,
        filename=image_data.filename,
        file_size=estimated_size,
        faces=faces,
        input_type="base64"
    )
    
    return FaceDetectionResponse(
        transaction_id=transaction_id,
        face_count=len(faces),
        faces=faces,
        success=True,
        message=f"Successfully detected {len(faces)} face(s) from base64 image"
    )


def _log_detection_and_return_id(
    db: Session,
    user: User,
    request: Request,
    filename: str,
    file_size: int,
    faces: list,
    input_type: str
) -> uuid.UUID:
    """Helper function to log face detection and return transaction ID"""
    transaction_id = uuid.uuid4()
    ip_address = request.client.host if request.client else None
    
    log_user_action(
        db,
        user.id,
        "face_detection",
        {
            "filename": filename,
            "file_size": file_size,
            "face_count": len(faces),
            "input_type": input_type,
            "faces": [face.dict() for face in faces]
        },
        ip_address
    )
    
    return transaction_id