from fastapi import APIRouter, Depends, HTTPException, status, Request
from sqlalchemy.orm import Session
from app.database import get_db
from app.schemas.user import UserCreate, UserLogin, UserResponse
from app.schemas.auth import Token
from app.services.auth_service import AuthService
from app.utils.auth import create_access_token

router = APIRouter(prefix="/api/auth", tags=["authentication"])


@router.post("/register", response_model=Token)
def register(
    user_data: UserCreate,
    request: Request,
    db: Session = Depends(get_db)
):
    auth_service = AuthService(db)
    ip_address = request.client.host if request.client else None
    
    user = auth_service.register_user(user_data, ip_address)
    access_token = create_access_token(data={"sub": str(user.id)})
    
    return Token(
        access_token=access_token,
        token_type="bearer",
        user=UserResponse.from_orm(user)
    )


@router.post("/login", response_model=Token)
def login(
    login_data: UserLogin,
    request: Request,
    db: Session = Depends(get_db)
):
    auth_service = AuthService(db)
    ip_address = request.client.host if request.client else None
    
    user = auth_service.authenticate_user(login_data, ip_address)
    access_token = create_access_token(data={"sub": str(user.id)})
    
    return Token(
        access_token=access_token,
        token_type="bearer",
        user=UserResponse.from_orm(user)
    )