from fastapi import FastAPI, HTTPException, Request
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse
from fastapi.security import HTTPBearer
from loguru import logger
import sys
from app.routers import auth_router, face_router, logs_router
from app.database import engine, Base

# Configure loguru
logger.remove()
logger.add(sys.stdout, level="INFO", format="{time} | {level} | {message}")

# Security scheme for Swagger UI
security = HTTPBearer()

app = FastAPI(
    title="Face Detection API",
    description="FastAPI application for face detection using YuNet model",
    version="1.0.0",
    docs_url="/docs",
    redoc_url="/redoc"
)

# CORS middleware  Farklı domain'lerden API'ye erişim kontrolü
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True, # JWT Token için gerekli (Cookie ve authentication header izin verir)
    allow_methods=["*"],
    allow_headers=["*"],
)

# Include routers
app.include_router(auth_router)
app.include_router(face_router)
app.include_router(logs_router)


# Exception handlers
@app.exception_handler(HTTPException)
async def http_exception_handler(request: Request, exc: HTTPException):
    logger.error(f"HTTP {exc.status_code}: {exc.detail}")
    return JSONResponse(
        status_code=exc.status_code,
        content={"detail": exc.detail, "status_code": exc.status_code}
    )


@app.exception_handler(Exception)
async def general_exception_handler(request: Request, exc: Exception):
    logger.error(f"Unhandled exception: {str(exc)}")
    return JSONResponse(
        status_code=500,
        content={"detail": "Internal server error", "status_code": 500}
    )


# Startup event
@app.on_event("startup")
def startup_event():
    logger.info("Starting Face Recognition API...")
    # Create database tables
    Base.metadata.create_all(bind=engine, checkfirst=True)
    logger.info("Database tables created successfully")


# Health check endpoint
@app.get("/health")
async def health_check():
    return {"status": "healthy", "message": "Face Recognition API is running"}


# Root endpoint
@app.get("/")
async def root():
    return {
        "message": "Welcome to Face Recognition API",
        "docs": "/docs",
        "redoc": "/redoc",
        "health": "/health"
    }

@app.get("/hello")
async def hello():
    return {"message": "Merhaba Dünya!"}

@app.get("/topla")
async def topla(a: float, b: float):
    return {"toplam": a+b}
