# FastAPI Face Recognition Application

A complete FastAPI application for face detection using OpenCV's YuNet model with JWT authentication, PostgreSQL database, and comprehensive logging.

## Features

- **Authentication**: JWT-based user registration and login
- **Face Detection**: OpenCV YuNet model for accurate face detection
- **Database**: PostgreSQL with SQLAlchemy async ORM
- **Logging**: Comprehensive transaction logging with unique IDs
- **Security**: Password hashing, CORS protection, input validation
- **Testing**: 80%+ test coverage with pytest
- **Docker**: Full containerization support
- **API Documentation**: Auto-generated Swagger/OpenAPI docs

## Project Structure

```
project/
├── app/
│   ├── __init__.py
│   ├── main.py              # FastAPI application entry point
│   ├── config.py            # Configuration settings
│   ├── database.py          # Database connection and session management
│   ├── models/              # SQLAlchemy models
│   │   ├── user.py
│   │   └── user_log.py
│   ├── schemas/             # Pydantic schemas
│   │   ├── user.py
│   │   ├── user_log.py
│   │   ├── face_detection.py
│   │   └── auth.py
│   ├── routers/             # API route handlers
│   │   ├── auth.py
│   │   ├── face_detection.py
│   │   └── logs.py
│   ├── services/            # Business logic
│   │   ├── auth_service.py
│   │   ├── face_detection_service.py
│   │   └── user_service.py
│   └── utils/               # Utility functions
│       ├── auth.py
│       └── logging.py
├── models/
│   └── yunet_model.onnx     # YuNet face detection model
├── tests/                   # Comprehensive test suite
├── .env.example             # Environment variables template
├── requirements.txt         # Python dependencies
├── Dockerfile              # Docker container configuration
├── docker-compose.yml      # Multi-container setup
└── README.md               # This file
```

## Installation

### Local Development

1. Create virtual environment:
```bash
python -m venv venv
source venv/bin/activate  # On Windows: venv\Scripts\activate
```

2. Install dependencies:
```bash
pip install -r requirements.txt
```

3. Set up environment variables:
```bash
cp .env.example .env
# Edit .env with your PostgreSQL connection string and other settings
```

4. Set up PostgreSQL database:
```bash
# Create database named 'facerecognition' in your PostgreSQL instance
# Update DATABASE_URL in .env file
```

5. Run the application:
```bash
uvicorn app.main:app --reload
```

## Environment Variables

Create a `.env` file with the following variables:

```env
DATABASE_URL=postgresql+asyncpg://username:password@localhost/dbname
JWT_SECRET_KEY=your-secret-key-here
JWT_ALGORITHM=HS256
ACCESS_TOKEN_EXPIRE_MINUTES=30
YUNET_MODEL_PATH=models/yunet_model.onnx
MAX_FILE_SIZE=10485760
ALLOWED_EXTENSIONS=jpg,jpeg,png
```

## API Endpoints

### Authentication

#### Register User
- **POST** `/api/auth/register`
- **Body**: `{"username": "string", "email": "string", "password": "string"}`
- **Response**: User info + JWT token

#### Login
- **POST** `/api/auth/login`
- **Body**: `{"email": "string", "password": "string"}`
- **Response**: User info + JWT token

### Face Detection

#### Detect Faces (File Upload)
- **POST** `/api/face/detect`
- **Headers**: `Authorization: Bearer <token>`
- **Body**: Multipart form with image file
- **Response**: Face detection results with coordinates and confidence scores

#### Detect Faces (Base64)
- **POST** `/api/face/detect-base64`
- **Headers**: `Authorization: Bearer <token>`
- **Body**: JSON with base64 encoded image
- **Response**: Face detection results with coordinates and confidence scores

### Logs

#### Get User Logs
- **GET** `/api/logs/user/{user_id}?limit=50&offset=0`
- **Headers**: `Authorization: Bearer <token>`
- **Response**: Paginated list of user transaction logs

### Health Check

#### Health Status
- **GET** `/health`
- **Response**: API health status

## Usage Examples

### 1. Register a new user

```bash
curl -X POST "http://localhost:8000/api/auth/register" \
  -H "Content-Type: application/json" \
  -d '{
    "username": "johndoe",
    "email": "john@example.com", 
    "password": "securepassword123"
  }'
```

### 2. Login

```bash
curl -X POST "http://localhost:8000/api/auth/login" \
  -H "Content-Type: application/json" \
  -d '{
    "email": "john@example.com",
    "password": "securepassword123"
  }'
```

### 3. Detect faces in an image (File Upload)

```bash
curl -X POST "http://localhost:8000/api/face/detect" \
  -H "Authorization: Bearer <your-jwt-token>" \
  -F "file=@path/to/your/image.jpg"
```

### 3b. Detect faces in an image (Base64)

```bash
# Convert image to base64 first
BASE64_IMAGE=$(base64 -w 0 path/to/your/image.jpg)

curl -X POST "http://localhost:8000/api/face/detect-base64" \
  -H "Authorization: Bearer <your-jwt-token>" \
  -H "Content-Type: application/json" \
  -d '{
    "image_data": "'$BASE64_IMAGE'",
    "filename": "image.jpg"
  }'
```

### 3c. Using Data URL format (Base64)

```bash
curl -X POST "http://localhost:8000/api/face/detect-base64" \
  -H "Authorization: Bearer <your-jwt-token>" \
  -H "Content-Type: application/json" \
  -d '{
    "image_data": "data:image/jpeg;base64,/9j/4AAQSkZJRgABAQEA...",
    "filename": "image.jpg"
  }'
```

### JavaScript Example (Base64)

```javascript
// Convert file to base64
function fileToBase64(file) {
    return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.readAsDataURL(file);
        reader.onload = () => resolve(reader.result);
        reader.onerror = error => reject(error);
    });
}

// Detect faces from file input
async function detectFacesFromFile(file, token) {
    const base64Data = await fileToBase64(file);
    
    const response = await fetch('/api/face/detect-base64', {
        method: 'POST',
        headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            image_data: base64Data, // Includes data URL prefix
            filename: file.name
        })
    });
    
    return response.json();
}

// Usage
const fileInput = document.getElementById('imageFile');
const file = fileInput.files[0];
const result = await detectFacesFromFile(file, 'your-jwt-token');
console.log(`Detected ${result.face_count} faces:`, result.faces);
```

### 4. Get user logs

```bash
curl -X GET "http://localhost:8000/api/logs/user/1?limit=10&offset=0" \
  -H "Authorization: Bearer <your-jwt-token>"
```

## Testing

Run the test suite:

```bash
# Run all tests
pytest

# Run with coverage
pytest --cov=app --cov-report=html

# Run specific test file
pytest tests/test_auth.py

# Run with verbose output
pytest -v
```

The test suite includes:
- Authentication tests (registration, login, password hashing)
- Face detection tests (with/without auth, file validation)
- Database CRUD tests
- API endpoint tests
- Logging functionality tests

## Database Schema

### Users Table
- `id` (Primary Key)
- `username` (unique, not null)
- `email` (unique, not null)
- `hashed_password` (not null)
- `created_at` (timestamp)
- `updated_at` (timestamp)

### UserLogs Table
- `id` (Primary Key)
- `transaction_id` (UUID, unique, indexed)
- `user_id` (Foreign Key -> Users)
- `action` (string: 'register', 'login', 'face_detection')
- `details` (JSON)
- `ip_address` (string, optional)
- `timestamp` (timestamp, default=now)

## Face Detection

The application uses OpenCV's YuNet model for face detection:
- **Model**: YuNet (face_detection_yunet_2023mar.onnx)
- **Confidence Threshold**: 0.6
- **Supported Formats**: JPG, JPEG, PNG, GIF, BMP, WEBP
- **Max File Size**: 10MB (configurable)
- **Input Methods**: File upload or Base64 encoded images
- **Output**: Face coordinates (x, y, width, height) + confidence score

### Input Methods

#### 1. File Upload (`/api/face/detect`)
- Standard multipart/form-data file upload
- Direct file validation and processing
- Supports all common image formats

#### 2. Base64 Input (`/api/face/detect-base64`)
- JSON payload with base64 encoded image data
- Supports both raw base64 and data URL format
- Automatic format detection and validation
- Example formats:
  - Raw base64: `"iVBORw0KGgoAAAANSUhEUgAA..."`
  - Data URL: `"data:image/jpeg;base64,/9j/4AAQSkZJRg..."`

### Base64 Features
- **Automatic Data URL Detection**: Strips `data:image/...;base64,` prefix automatically
- **Size Validation**: Estimates file size from base64 length
- **Format Validation**: Validates image format after decoding
- **Error Handling**: Comprehensive error messages for invalid data

## Security Features

- Password hashing with bcrypt
- JWT token-based authentication
- Input validation with Pydantic
- File upload validation (type, size)
- CORS protection
- SQL injection prevention with SQLAlchemy
- User isolation (users can only access their own data)

## Production Deployment

### Environment Setup
1. Use strong JWT secret keys
2. Configure proper CORS origins
3. Use production PostgreSQL database
4. Set up proper logging levels
5. Configure reverse proxy (nginx)
6. Use HTTPS certificates

### Docker Production
```bash
# Build for production
docker-compose -f docker-compose.yml up -d

# Scale the application
docker-compose up --scale app=3
```

## API Documentation

Once the application is running, visit:
- **Swagger UI**: `http://localhost:8000/docs`
- **ReDoc**: `http://localhost:8000/redoc`
- **OpenAPI JSON**: `http://localhost:8000/openapi.json`

## Contributing

1. Fork the repository
2. Create a feature branch
3. Add tests for new functionality
4. Ensure all tests pass
5. Submit a pull request

## License

This project is licensed under the MIT License - see the LICENSE file for details.

## Support

For support or questions, please open an issue in the repository.